<?php
include 'db.php'; // Ensure this file correctly establishes $conn

// Get start and end date from query parameters
// Basic validation: check if parameters are set
if (!isset($_GET['startDate']) || !isset($_GET['endDate'])) {
    header("HTTP/1.1 400 Bad Request");
    echo "Error: Start date and end date parameters are required.";
    exit;
}

$startDate = $_GET['startDate'];
$endDate = $_GET['endDate'];

// Validate date format (YYYY-MM-DD)
function validateDate($date, $format = 'Y-m-d') {
    $d = DateTime::createFromFormat($format, $date);
    return $d && $d->format($format) === $date;
}

if (!validateDate($startDate) || !validateDate($endDate)) {
    header("HTTP/1.1 400 Bad Request");
    echo "Error: Invalid date format. Please use YYYY-MM-DD.";
    exit;
}

// To prevent SQL injection, even without prepared statements, escape the input.
// This is crucial when not using prepared statements.
$escapedStartDate = mysqli_real_escape_string($conn, $startDate);
$escapedEndDate = mysqli_real_escape_string($conn, $endDate);

// Adjust end date to include the whole day if 'date' column is DATETIME
// If 'date' column is just DATE, this adjustment might not be strictly needed
// but DATE(date_column) <= '$escapedEndDate' is safer.
// For DATETIME, to include records up to 23:59:59 of the end date:
// One way is to use DATE() function in SQL or adjust the end date.
// Let's use DATE() in SQL for clarity assuming 'date' column might be DATETIME.

// Query to get subscription data
// Using DATE(date) allows comparison with 'YYYY-MM-DD' strings directly.
$query = "SELECT id, email, date FROM Subscribe 
          WHERE delete_status = 0 
          AND DATE(date) >= '$escapedStartDate' 
          AND DATE(date) <= '$escapedEndDate' 
          ORDER BY date DESC";

$result = mysqli_query($conn, $query);

if (!$result) {
    header("HTTP/1.1 500 Internal Server Error");
    // In a production environment, you might log the detailed error
    // and show a generic message to the user.
    echo "Error executing query: " . mysqli_error($conn);
    exit;
}

// Count total subscriptions for the selected period
$countQuery = "SELECT COUNT(*) as total FROM Subscribe 
               WHERE delete_status = 0 
               AND DATE(date) >= '$escapedStartDate' 
               AND DATE(date) <= '$escapedEndDate'";
$countResult = mysqli_query($conn, $countQuery);

if (!$countResult) {
    header("HTTP/1.1 500 Internal Server Error");
    echo "Error executing count query: " . mysqli_error($conn);
    exit;
}
$countRow = mysqli_fetch_assoc($countResult);
$totalSubscriptions = $countRow['total'];

// Create CSV content
$csvContent = "ID,Email,Subscription Date\n"; // CSV header

while ($row = mysqli_fetch_assoc($result)) {
    // Sanitize data going into CSV to prevent CSV injection, if necessary
    // For simple email and date, direct output is often okay, but be mindful.
    $csvContent .= '"' . str_replace('"', '""', $row['id']) . '","' . 
                   str_replace('"', '""', $row['email']) . '","' . 
                   str_replace('"', '""', $row['date']) . '"' . "\n";
}

$csvContent .= "\nTotal Subscriptions:," . $totalSubscriptions . "\n";

// Sanitize filename parts
$safeStartDate = preg_replace('/[^A-Za-z0-9_-]/', '', $startDate);
$safeEndDate = preg_replace('/[^A-Za-z0-9_-]/', '', $endDate);
$filename = "subscription_report_" . $safeStartDate . "_to_" . $safeEndDate . ".csv";

// Set headers for CSV download
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="' . $filename . '"');

// Output CSV content
echo $csvContent;
mysqli_close($conn);
exit;
?>